<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isStaff()) {
    setFlashMessage('error', 'Access denied. Staff privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();
$userId = getCurrentUserId();
$departmentId = $_SESSION['department_id'] ?? null;

// Get stats for assigned complaints
$stats = [];

// Assigned to me
$stmt = $db->prepare("SELECT COUNT(*) as total FROM complaints WHERE assigned_to = ?");
$stmt->execute([$userId]);
$stats['assigned'] = $stmt->fetch()['total'];

// Open assigned
$stmt = $db->prepare("SELECT COUNT(*) as total FROM complaints WHERE assigned_to = ? AND status IN ('open', 'in_progress', 'pending_customer')");
$stmt->execute([$userId]);
$stats['open'] = $stmt->fetch()['total'];

// Resolved by me
$stmt = $db->prepare("SELECT COUNT(*) as total FROM complaints WHERE assigned_to = ? AND status IN ('resolved', 'closed')");
$stmt->execute([$userId]);
$stats['resolved'] = $stmt->fetch()['total'];

// Department complaints (if staff has department)
if ($departmentId) {
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM complaints WHERE department_id = ? AND status NOT IN ('resolved', 'closed')");
    $stmt->execute([$departmentId]);
    $stats['department'] = $stmt->fetch()['total'];
} else {
    $stats['department'] = 0;
}

// Get assigned complaints
$assignedComplaints = $db->prepare("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp, 
           i.title as issue_title, d.name as department_name, b.name as branch_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    WHERE c.assigned_to = ?
    ORDER BY 
        CASE c.priority 
            WHEN 'urgent' THEN 1 
            WHEN 'high' THEN 2 
            WHEN 'medium' THEN 3 
            ELSE 4 
        END,
        c.created_at DESC
    LIMIT 10
");
$assignedComplaints->execute([$userId]);
$assignedComplaints = $assignedComplaints->fetchAll();

// Get unassigned complaints in user's department
$unassignedComplaints = $db->prepare("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp, 
           i.title as issue_title, b.name as branch_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    WHERE c.department_id = ? AND c.assigned_to IS NULL AND c.status NOT IN ('resolved', 'closed')
    ORDER BY c.created_at ASC
    LIMIT 5
");
$unassignedComplaints->execute([$departmentId]);
$unassignedComplaints = $unassignedComplaints->fetchAll();


$pageTitle = 'Staff Dashboard - Support Center';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-8">
    <h1 class="text-3xl font-bold text-gray-800">Staff Dashboard</h1>
    <p class="text-gray-500 mt-1">Welcome back, <?= sanitize($_SESSION['user_name']) ?></p>
</div>

<!-- Stats Cards -->
<div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-blue-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Assigned to Me</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['assigned'] ?></p>
            </div>
            <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <i class="fas fa-tasks text-blue-500 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-yellow-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Open Cases</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['open'] ?></p>
            </div>
            <div class="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
                <i class="fas fa-clock text-yellow-500 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-green-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Resolved</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['resolved'] ?></p>
            </div>
            <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                <i class="fas fa-check-circle text-green-500 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-purple-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Department Queue</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['department'] ?></p>
            </div>
            <div class="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                <i class="fas fa-building text-purple-500 text-xl"></i>
            </div>
        </div>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
    <!-- My Assigned Complaints -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-lg font-semibold text-gray-800">My Assigned Complaints</h2>
            <a href="<?= BASE_URL ?>/staff/complaints.php" class="text-primary hover:underline text-sm">View All</a>
        </div>

        <?php if (empty($assignedComplaints)): ?>
        <div class="text-center py-8 text-gray-500">
            <i class="fas fa-inbox text-4xl mb-3"></i>
            <p>No complaints assigned to you</p>
        </div>
        <?php else: ?>
        <div class="space-y-4">
            <?php foreach ($assignedComplaints as $complaint): ?>
            <a href="<?= BASE_URL ?>/complaint/view.php?ticket=<?= sanitize($complaint['ticket_number']) ?>" 
               class="block p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition">
                <div class="flex items-start justify-between">
                    <div class="flex-1">
                        <div class="flex items-center gap-2 mb-1">
                            <span class="px-2 py-0.5 rounded-full text-xs font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                                <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                            </span>
                            <span class="px-2 py-0.5 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                                <?= ucfirst($complaint['priority']) ?>
                            </span>
                        </div>
                        <p class="font-medium text-gray-800 line-clamp-1"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></p>
                        <p class="text-sm text-gray-500 mt-1">
                            <?= sanitize($complaint['branch_name']) ?> • <?= formatDate($complaint['created_at']) ?>
                        </p>
                    </div>
                    <i class="fas fa-chevron-right text-gray-400"></i>
                </div>
            </a>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>

    <!-- Unassigned Department Complaints -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-lg font-semibold text-gray-800">Unassigned in Department</h2>
        </div>

        <?php if (empty($unassignedComplaints)): ?>
        <div class="text-center py-8 text-gray-500">
            <i class="fas fa-check-circle text-4xl mb-3 text-green-500"></i>
            <p>All complaints are assigned!</p>
        </div>
        <?php else: ?>
        <div class="space-y-4">
            <?php foreach ($unassignedComplaints as $complaint): ?>
            <div class="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                <div class="flex items-start justify-between">
                    <div class="flex-1">
                        <div class="flex items-center gap-2 mb-1">
                            <span class="px-2 py-0.5 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                                <?= ucfirst($complaint['priority']) ?>
                            </span>
                            <span class="text-xs text-gray-500"><?= sanitize($complaint['ticket_number']) ?></span>
                        </div>
                        <p class="font-medium text-gray-800 line-clamp-1"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></p>
                        <p class="text-sm text-gray-500 mt-1"><?= sanitize($complaint['branch_name']) ?></p>
                    </div>
                    <form method="POST" action="<?= BASE_URL ?>/staff/assign.php">
                        <input type="hidden" name="complaint_id" value="<?= $complaint['id'] ?>">
                        <button type="submit" class="px-3 py-1 bg-primary text-white rounded text-sm hover:opacity-90 transition">
                            Take
                        </button>
                    </form>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
