<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isBranchManager()) {
    setFlashMessage('error', 'Access denied. Branch manager privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();
$userId = getCurrentUserId();
$branchId = getCurrentUserBranchId();

// Get branch info
$branchStmt = $db->prepare("SELECT * FROM branches WHERE id = ?");
$branchStmt->execute([$branchId]);
$branch = $branchStmt->fetch();

// Filters
$statusFilter = $_GET['status'] ?? '';
$priorityFilter = $_GET['priority'] ?? '';
$search = $_GET['search'] ?? '';

$where = "WHERE c.branch_id = ?";
$params = [$branchId];

if ($statusFilter) {
    $where .= " AND c.status = ?";
    $params[] = $statusFilter;
}

if ($priorityFilter) {
    $where .= " AND c.priority = ?";
    $params[] = $priorityFilter;
}

if ($search) {
    $where .= " AND (c.ticket_number LIKE ? OR o.order_number LIKE ? OR u.whatsapp_number LIKE ?)";
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

// Pagination
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 15;
$offset = ($page - 1) * $perPage;

// Get total count
$countStmt = $db->prepare("
    SELECT COUNT(*) as total 
    FROM complaints c 
    JOIN orders o ON c.order_id = o.id 
    JOIN users u ON c.customer_id = u.id
    $where
");
$countStmt->execute($params);
$totalCount = $countStmt->fetch()['total'];
$totalPages = ceil($totalCount / $perPage);

// Get complaints
$stmt = $db->prepare("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp,
           i.title as issue_title, d.name as department_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    $where
    ORDER BY 
        CASE c.priority 
            WHEN 'urgent' THEN 1 
            WHEN 'high' THEN 2 
            WHEN 'medium' THEN 3 
            ELSE 4 
        END,
        c.created_at DESC
    LIMIT $perPage OFFSET $offset
");
$stmt->execute($params);
$complaints = $stmt->fetchAll();

$pageTitle = 'Branch Complaints - ' . ($branch['name'] ?? 'Support Center');
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6">
    <div class="flex items-center justify-between flex-wrap gap-4">
        <div>
            <h1 class="text-2xl font-bold text-gray-800">Branch Complaints</h1>
            <p class="text-gray-500 flex items-center gap-2 mt-1">
                <i class="fas fa-store"></i>
                <?= sanitize($branch['name'] ?? 'Unknown Branch') ?> • <?= $totalCount ?> complaints
            </p>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="bg-white rounded-xl shadow-lg p-4 mb-6">
    <form method="GET" class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
        <div>
            <input type="text" name="search" placeholder="Search ticket, order, phone..." 
                value="<?= sanitize($search) ?>"
                class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
        </div>
        <div>
            <select name="status" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                <option value="">All Status</option>
                <option value="open" <?= $statusFilter === 'open' ? 'selected' : '' ?>>Open</option>
                <option value="in_progress" <?= $statusFilter === 'in_progress' ? 'selected' : '' ?>>In Progress</option>
                <option value="pending_customer" <?= $statusFilter === 'pending_customer' ? 'selected' : '' ?>>Pending Customer</option>
                <option value="resolved" <?= $statusFilter === 'resolved' ? 'selected' : '' ?>>Resolved</option>
                <option value="closed" <?= $statusFilter === 'closed' ? 'selected' : '' ?>>Closed</option>
            </select>
        </div>
        <div>
            <select name="priority" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent">
                <option value="">All Priority</option>
                <option value="urgent" <?= $priorityFilter === 'urgent' ? 'selected' : '' ?>>Urgent</option>
                <option value="high" <?= $priorityFilter === 'high' ? 'selected' : '' ?>>High</option>
                <option value="medium" <?= $priorityFilter === 'medium' ? 'selected' : '' ?>>Medium</option>
                <option value="low" <?= $priorityFilter === 'low' ? 'selected' : '' ?>>Low</option>
            </select>
        </div>
        <div class="flex gap-2">
            <button type="submit" class="flex-1 gradient-bg text-white px-4 py-2 rounded-lg font-medium hover:opacity-90 transition">
                <i class="fas fa-search mr-1"></i> Filter
            </button>
            <a href="<?= BASE_URL ?>/manager/complaints.php" class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition">
                <i class="fas fa-times"></i>
            </a>
        </div>
    </form>
</div>

<!-- Complaints Table -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <?php if (empty($complaints)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-inbox text-5xl mb-4"></i>
        <p class="text-lg">No complaints found</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ticket</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Issue</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Customer</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Priority</th>
                    <th class="px-4 md:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden xl:table-cell">Date</th>
                    <th class="px-4 md:px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($complaints as $complaint): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-4 md:px-6 py-4">
                        <span class="font-mono text-sm"><?= sanitize($complaint['ticket_number']) ?></span>
                        <p class="text-xs text-gray-500 mt-1"><?= sanitize($complaint['order_number']) ?></p>
                    </td>
                    <td class="px-4 md:px-6 py-4 hidden md:table-cell">
                        <span class="text-gray-800 line-clamp-1"><?= sanitize($complaint['issue_title'] ?? 'N/A') ?></span>
                    </td>
                    <td class="px-4 md:px-6 py-4">
                        <a href="https://wa.me/<?= preg_replace('/[^0-9]/', '', $complaint['customer_whatsapp']) ?>" 
                           target="_blank" class="text-green-600 hover:underline flex items-center gap-1">
                            <i class="fab fa-whatsapp"></i>
                            <span class="hidden sm:inline"><?= sanitize($complaint['customer_whatsapp']) ?></span>
                            <span class="sm:hidden">Chat</span>
                        </a>
                    </td>
                    <td class="px-4 md:px-6 py-4">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                            <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                        </span>
                    </td>
                    <td class="px-4 md:px-6 py-4 hidden lg:table-cell">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                            <?= ucfirst($complaint['priority']) ?>
                        </span>
                    </td>
                    <td class="px-4 md:px-6 py-4 hidden xl:table-cell text-sm text-gray-500">
                        <?= formatDate($complaint['created_at']) ?>
                    </td>
                    <td class="px-4 md:px-6 py-4 text-right">
                        <a href="<?= BASE_URL ?>/complaint/view.php?ticket=<?= sanitize($complaint['ticket_number']) ?>" 
                           class="text-primary hover:underline">
                            <i class="fas fa-eye"></i>
                            <span class="hidden sm:inline ml-1">View</span>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="px-4 md:px-6 py-4 border-t flex flex-wrap items-center justify-between gap-4">
        <p class="text-sm text-gray-500">
            Showing <?= $offset + 1 ?> to <?= min($offset + $perPage, $totalCount) ?> of <?= $totalCount ?>
        </p>
        <div class="flex gap-2">
            <?php if ($page > 1): ?>
            <a href="?page=<?= $page - 1 ?>&status=<?= $statusFilter ?>&priority=<?= $priorityFilter ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50">Previous</a>
            <?php endif; ?>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
            <a href="?page=<?= $i ?>&status=<?= $statusFilter ?>&priority=<?= $priorityFilter ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded <?= $i === $page ? 'bg-primary text-white' : 'hover:bg-gray-50' ?>">
                <?= $i ?>
            </a>
            <?php endfor; ?>
            
            <?php if ($page < $totalPages): ?>
            <a href="?page=<?= $page + 1 ?>&status=<?= $statusFilter ?>&priority=<?= $priorityFilter ?>&search=<?= urlencode($search) ?>" 
               class="px-3 py-1 border rounded hover:bg-gray-50">Next</a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
