-- Migration Script v2: Add Branch Manager Role, Access Tokens, and WhatsApp Message Logging
-- Run this script if you have an existing database from v1

-- Step 1: Modify users table to add branch_id and update role enum
ALTER TABLE users 
    ADD COLUMN branch_id INT NULL AFTER role,
    ADD FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE SET NULL;

-- Update role enum to include branch_manager
ALTER TABLE users 
    MODIFY COLUMN role ENUM('admin', 'branch_manager', 'staff', 'customer') DEFAULT 'customer';

-- Step 2: Add access_token to complaints table
ALTER TABLE complaints 
    ADD COLUMN access_token VARCHAR(64) NOT NULL UNIQUE AFTER ticket_number;

-- Generate access tokens for existing complaints
UPDATE complaints SET access_token = MD5(CONCAT(id, ticket_number, NOW(), RAND())) WHERE access_token = '' OR access_token IS NULL;

-- Step 3: Make order_id unique to prevent duplicate complaints per order
-- Note: This may fail if you have duplicate complaints for the same order
-- You'll need to resolve duplicates first
ALTER TABLE complaints 
    ADD UNIQUE INDEX unique_order_complaint (order_id);

-- Step 4: Create WhatsApp message log table
CREATE TABLE IF NOT EXISTS whatsapp_messages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    complaint_id INT NULL,
    recipient_number VARCHAR(20) NOT NULL,
    message_type ENUM('otp', 'complaint_receipt', 'manager_notification', 'customer_update', 'other') NOT NULL,
    message_content TEXT NOT NULL,
    sent_by INT NULL,
    is_sent BOOLEAN DEFAULT FALSE,
    sent_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (complaint_id) REFERENCES complaints(id) ON DELETE SET NULL,
    FOREIGN KEY (sent_by) REFERENCES users(id) ON DELETE SET NULL
);

-- Step 5: Insert sample branch managers (password: password)
-- You should update these with real branch managers
INSERT INTO users (email, password, full_name, whatsapp_number, role, branch_id, is_verified) VALUES
('manager.main@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Main Branch Manager', '+1234567801', 'branch_manager', 1, TRUE),
('manager.north@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'North Branch Manager', '+1234567802', 'branch_manager', 2, TRUE),
('manager.south@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'South Branch Manager', '+1234567803', 'branch_manager', 3, TRUE),
('manager.east@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'East Mall Manager', '+1234567804', 'branch_manager', 4, TRUE),
('manager.west@company.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'West Plaza Manager', '+1234567805', 'branch_manager', 5, TRUE)
ON DUPLICATE KEY UPDATE role = 'branch_manager';

-- Step 6: Update existing staff to have branch_id (assign to branch 1 by default)
UPDATE users SET branch_id = 1 WHERE role = 'staff' AND branch_id IS NULL;

-- Done!
SELECT 'Migration v2 completed successfully!' as status;
