<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn()) {
    redirect('/auth/login.php');
}

$db = getDB();
$userId = getCurrentUserId();

// Get all orders for this customer
$stmt = $db->prepare("
    SELECT o.*, d.name as department_name,
           (SELECT COUNT(*) FROM complaints WHERE order_id = o.id) as complaint_count
    FROM orders o
    LEFT JOIN departments d ON o.department_id = d.id
    WHERE o.customer_id = ?
    ORDER BY o.created_at DESC
");
$stmt->execute([$userId]);
$orders = $stmt->fetchAll();

$pageTitle = 'My Orders - Support Center';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-6">
    <h1 class="text-2xl font-bold text-gray-800">My Orders</h1>
    <p class="text-gray-500">View your orders and submit complaints</p>
</div>

<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <?php if (empty($orders)): ?>
    <div class="text-center py-12 text-gray-500">
        <i class="fas fa-shopping-cart text-5xl mb-4"></i>
        <p class="text-lg">No orders found</p>
        <p class="text-sm mt-2">You don't have any orders yet.</p>
    </div>
    <?php else: ?>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Order #</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden md:table-cell">Complaints</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($orders as $order): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="font-mono font-medium text-gray-800"><?= sanitize($order['order_number']) ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-gray-600">
                        <?= date('M d, Y', strtotime($order['order_date'])) ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap font-medium text-gray-800">
                        $<?= number_format($order['total_amount'], 2) ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php
                        $statusColors = [
                            'pending' => 'bg-yellow-100 text-yellow-800',
                            'processing' => 'bg-blue-100 text-blue-800',
                            'shipped' => 'bg-purple-100 text-purple-800',
                            'delivered' => 'bg-green-100 text-green-800',
                            'cancelled' => 'bg-red-100 text-red-800'
                        ];
                        $statusColor = $statusColors[$order['status']] ?? 'bg-gray-100 text-gray-800';
                        ?>
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= $statusColor ?> capitalize">
                            <?= sanitize($order['status']) ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                        <?php if ($order['complaint_count'] > 0): ?>
                        <span class="text-sm text-gray-600"><?= $order['complaint_count'] ?> complaint(s)</span>
                        <?php else: ?>
                        <span class="text-sm text-gray-400">None</span>
                        <?php endif; ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-right">
                        <a href="<?= BASE_URL ?>/complaint/submit.php?order=<?= urlencode($order['order_number']) ?>" 
                           class="inline-flex items-center px-3 py-1.5 bg-primary text-white rounded-lg text-sm hover:opacity-90 transition">
                            <i class="fas fa-plus mr-1"></i>New Complaint
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
