<?php
/**
 * WhatsApp API Service Configuration
 * 
 * Uses the standalone WhatsApp API Service for sending messages
 * 
 * Setup:
 * 1. Deploy WhatsApp API Service on your VPS
 * 2. Create an API key from the web interface
 * 3. Update WHATSAPP_API_KEY below
 */

// WhatsApp API Service Configuration
// For local development: http://localhost:3001
// For production: http://YOUR_VPS_IP:3001
define('WHATSAPP_SERVICE_URL', 'http://158.220.103.97:3001/');
define('WHATSAPP_API_KEY', 'wa_36dbc2bc8e0b4e0fa0355e1b758ea803'); // Get this from WhatsApp API Service dashboard
define('WHATSAPP_SERVICE_TIMEOUT', 30); // seconds

// OTP Settings
define('OTP_LENGTH', 6);
define('OTP_EXPIRY_MINUTES', 5);
define('OTP_MAX_ATTEMPTS', 3);

class WhatsAppOTP {
    
    /**
     * Generate a random OTP code
     */
    public static function generateOTP() {
        return str_pad(random_int(0, pow(10, OTP_LENGTH) - 1), OTP_LENGTH, '0', STR_PAD_LEFT);
    }
    
    /**
     * Format phone number to international format
     */
    public static function formatPhoneNumber($phone) {
        // Remove all non-numeric characters except +
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        
        // If doesn't start with +, assume it needs country code
        if (strpos($phone, '+') !== 0) {
            // You can set a default country code here
            // For now, we'll just add + if missing
            if (strlen($phone) >= 10) {
                $phone = '+' . $phone;
            }
        }
        
        return $phone;
    }
    
    /**
     * Get API headers for authenticated requests
     */
    private static function getApiHeaders() {
        $headers = ['Content-Type: application/json'];
        if (defined('WHATSAPP_API_KEY') && WHATSAPP_API_KEY) {
            $headers[] = 'X-API-Key: ' . WHATSAPP_API_KEY;
        }
        return $headers;
    }

    /**
     * Check if WhatsApp service is connected
     */
    public static function checkServiceStatus() {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, WHATSAPP_SERVICE_URL . '/health');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            return [
                'status' => $data['status'] ?? 'unknown',
                'connected' => $data['whatsapp']['connected'] ?? ($data['connected'] ?? false)
            ];
        }
        
        return ['status' => 'offline', 'connected' => false];
    }

    /**
     * Send OTP via WhatsApp API Service
     */
    public static function sendOTP($phoneNumber, $otpCode) {
        // Check if WhatsApp service is running
        $status = self::checkServiceStatus();
        
        if (!isset($status['status']) || $status['status'] !== 'ok') {
            // Service not running - use dev mode
            return [
                'success' => true,
                'dev_mode' => true,
                'dev_otp' => $otpCode,
                'message' => 'DEV MODE: WhatsApp service not running. OTP: ' . $otpCode
            ];
        }
        
        if (!$status['connected']) {
            // Service running but not connected to WhatsApp
            return [
                'success' => true,
                'dev_mode' => true,
                'dev_otp' => $otpCode,
                'message' => 'DEV MODE: WhatsApp not connected. Please scan QR code. OTP: ' . $otpCode
            ];
        }

        // Send OTP via WhatsApp API Service
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, WHATSAPP_SERVICE_URL . '/api/send-otp');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'phone' => $phoneNumber,
            'otp' => $otpCode
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, self::getApiHeaders());
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, WHATSAPP_SERVICE_TIMEOUT);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            // Fallback to dev mode on connection error
            return [
                'success' => true,
                'dev_mode' => true,
                'dev_otp' => $otpCode,
                'message' => 'DEV MODE: Service connection error. OTP: ' . $otpCode
            ];
        }

        $result = json_decode($response, true);
        
        if ($httpCode === 200 && isset($result['success']) && $result['success']) {
            return ['success' => true];
        }

        // If service returns error, fallback to dev mode
        return [
            'success' => true,
            'dev_mode' => true,
            'dev_otp' => $otpCode,
            'message' => 'DEV MODE: ' . ($result['error'] ?? 'Failed to send') . '. OTP: ' . $otpCode
        ];
    }

    /**
     * Send custom WhatsApp message
     */
    public static function sendMessage($phoneNumber, $message) {
        $status = self::checkServiceStatus();
        
        if (!isset($status['status']) || $status['status'] !== 'ok' || !$status['connected']) {
            return ['success' => false, 'error' => 'WhatsApp service not available'];
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, WHATSAPP_SERVICE_URL . '/api/send');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'phone' => $phoneNumber,
            'message' => $message
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, self::getApiHeaders());
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, WHATSAPP_SERVICE_TIMEOUT);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $result = json_decode($response, true);
        
        if ($httpCode === 200 && isset($result['success']) && $result['success']) {
            return ['success' => true];
        }

        return [
            'success' => false,
            'error' => $result['error'] ?? 'Failed to send message'
        ];
    }
    
    /**
     * Store OTP in database
     */
    public static function storeOTP($db, $whatsappNumber, $otpCode) {
        $formattedNumber = self::formatPhoneNumber($whatsappNumber);
        $timezone = new DateTimeZone('Asia/Karachi');
		$expiresAt = (new DateTime('now', $timezone))->modify('+' . OTP_EXPIRY_MINUTES . ' minutes')->format('Y-m-d H:i:s');

        
        // Invalidate any existing OTPs for this number
        $stmt = $db->prepare("UPDATE otp_verifications SET is_used = 1 WHERE whatsapp_number = ? AND is_used = 0");
        $stmt->execute([$formattedNumber]);
        
        // Store new OTP
        $stmt = $db->prepare("INSERT INTO otp_verifications (whatsapp_number, otp_code, expires_at) VALUES (?, ?, ?)");
        return $stmt->execute([$formattedNumber, $otpCode, $expiresAt]);
    }
    
    /**
     * Verify OTP from database
     */
    public static function verifyOTP($db, $whatsappNumber, $otpCode) {
        $formattedNumber = self::formatPhoneNumber($whatsappNumber);
        
        $stmt = $db->prepare("
            SELECT * FROM otp_verifications 
            WHERE whatsapp_number = ? 
            AND otp_code = ? 
            AND is_used = 0 
            AND expires_at > NOW()
            ORDER BY created_at DESC 
            LIMIT 1
        ");
        $stmt->execute([$formattedNumber, $otpCode]);
        $otp = $stmt->fetch();
        
        if ($otp) {
            // Mark OTP as used
            $updateStmt = $db->prepare("UPDATE otp_verifications SET is_used = 1 WHERE id = ?");
            $updateStmt->execute([$otp['id']]);
            return ['success' => true];
        }
        
        return ['success' => false, 'error' => 'Invalid or expired OTP'];
    }
    
    /**
     * Get or create customer by WhatsApp number
     */
    public static function getOrCreateCustomer($db, $whatsappNumber, $email = null) {
        $formattedNumber = self::formatPhoneNumber($whatsappNumber);
        
        // Check if customer exists
        $stmt = $db->prepare("SELECT * FROM users WHERE whatsapp_number = ? AND role = 'customer'");
        $stmt->execute([$formattedNumber]);
        $customer = $stmt->fetch();
        
        if ($customer) {
            // Update email if provided and different
            if ($email && $email !== $customer['email']) {
                $updateStmt = $db->prepare("UPDATE users SET email = ?, is_verified = 1 WHERE id = ?");
                $updateStmt->execute([$email, $customer['id']]);
                $customer['email'] = $email;
            }
            return $customer;
        }
        
        // Create new customer
        $stmt = $db->prepare("INSERT INTO users (whatsapp_number, email, role, is_verified) VALUES (?, ?, 'customer', 1)");
        $stmt->execute([$formattedNumber, $email]);
        
        $customerId = $db->lastInsertId();
        
        return [
            'id' => $customerId,
            'whatsapp_number' => $formattedNumber,
            'email' => $email,
            'role' => 'customer',
            'is_verified' => 1
        ];
    }
}
