<?php
require_once __DIR__ . '/../config/config.php';

if (!isLoggedIn() || !isAdmin()) {
    setFlashMessage('error', 'Access denied. Admin privileges required.');
    redirect('/auth/login.php');
}

$db = getDB();

// Get stats
$stats = [];

// Total complaints
$stats['total'] = $db->query("SELECT COUNT(*) as total FROM complaints")->fetch()['total'];

// Open complaints
$stats['open'] = $db->query("SELECT COUNT(*) as total FROM complaints WHERE status IN ('open', 'in_progress', 'pending_customer')")->fetch()['total'];

// Resolved today
$stats['resolved_today'] = $db->query("SELECT COUNT(*) as total FROM complaints WHERE status IN ('resolved', 'closed') AND DATE(updated_at) = CURDATE()")->fetch()['total'];

// Total users
$stats['users'] = $db->query("SELECT COUNT(*) as total FROM users WHERE role = 'customer'")->fetch()['total'];

// Complaints by status
$statusStats = $db->query("
    SELECT status, COUNT(*) as count 
    FROM complaints 
    GROUP BY status
")->fetchAll();

// Complaints by department
$deptStats = $db->query("
    SELECT d.name, COUNT(c.id) as count
    FROM departments d
    LEFT JOIN complaints c ON d.id = c.department_id
    GROUP BY d.id, d.name
    ORDER BY count DESC
")->fetchAll();

// Recent complaints
$recentComplaints = $db->query("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp, 
           i.title as issue_title, d.name as department_name, b.name as branch_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    LEFT JOIN departments d ON c.department_id = d.id
    ORDER BY c.created_at DESC
    LIMIT 10
")->fetchAll();

// Urgent complaints
$urgentComplaints = $db->query("
    SELECT c.*, o.order_number, u.whatsapp_number as customer_whatsapp, 
           i.title as issue_title, b.name as branch_name
    FROM complaints c
    JOIN orders o ON c.order_id = o.id
    JOIN users u ON c.customer_id = u.id
    JOIN branches b ON c.branch_id = b.id
    LEFT JOIN complaint_issues i ON c.issue_id = i.id
    WHERE c.priority = 'urgent' AND c.status NOT IN ('resolved', 'closed')
    ORDER BY c.created_at ASC
    LIMIT 5
")->fetchAll();

// Complaints by branch
$branchStats = $db->query("
    SELECT b.name, COUNT(c.id) as count
    FROM branches b
    LEFT JOIN complaints c ON b.id = c.branch_id
    GROUP BY b.id, b.name
    ORDER BY count DESC
")->fetchAll();

$pageTitle = 'Admin Dashboard - Support Center';
require_once __DIR__ . '/../includes/header.php';
?>

<div class="mb-8">
    <h1 class="text-3xl font-bold text-gray-800">Admin Dashboard</h1>
    <p class="text-gray-500 mt-1">Overview of complaint management system</p>
</div>

<!-- Stats Cards -->
<div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-blue-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Total Complaints</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['total'] ?></p>
            </div>
            <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                <i class="fas fa-ticket-alt text-blue-500 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-yellow-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Open Complaints</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['open'] ?></p>
            </div>
            <div class="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
                <i class="fas fa-clock text-yellow-500 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-green-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Resolved Today</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['resolved_today'] ?></p>
            </div>
            <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center">
                <i class="fas fa-check-circle text-green-500 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-xl shadow-lg p-6 border-l-4 border-purple-500">
        <div class="flex items-center justify-between">
            <div>
                <p class="text-sm text-gray-500">Total Customers</p>
                <p class="text-3xl font-bold text-gray-800"><?= $stats['users'] ?></p>
            </div>
            <div class="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center">
                <i class="fas fa-users text-purple-500 text-xl"></i>
            </div>
        </div>
    </div>
</div>

<div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
    <!-- Status Distribution -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <h2 class="text-lg font-semibold text-gray-800 mb-4">Status Distribution</h2>
        <div class="space-y-3">
            <?php foreach ($statusStats as $stat): ?>
            <?php
            $percentage = $stats['total'] > 0 ? ($stat['count'] / $stats['total']) * 100 : 0;
            $colors = [
                'open' => 'bg-blue-500',
                'in_progress' => 'bg-yellow-500',
                'pending_customer' => 'bg-purple-500',
                'resolved' => 'bg-green-500',
                'closed' => 'bg-gray-500'
            ];
            $color = $colors[$stat['status']] ?? 'bg-gray-500';
            ?>
            <div>
                <div class="flex justify-between text-sm mb-1">
                    <span class="capitalize"><?= str_replace('_', ' ', $stat['status']) ?></span>
                    <span class="font-medium"><?= $stat['count'] ?></span>
                </div>
                <div class="w-full bg-gray-200 rounded-full h-2">
                    <div class="<?= $color ?> h-2 rounded-full" style="width: <?= $percentage ?>%"></div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Department Distribution -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <h2 class="text-lg font-semibold text-gray-800 mb-4">By Department</h2>
        <div class="space-y-3">
            <?php foreach ($deptStats as $dept): ?>
            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <span class="text-gray-700"><?= sanitize($dept['name']) ?></span>
                <span class="px-3 py-1 bg-primary text-white rounded-full text-sm font-medium"><?= $dept['count'] ?></span>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <!-- Urgent Complaints -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <h2 class="text-lg font-semibold text-gray-800 mb-4">
            <i class="fas fa-exclamation-triangle text-red-500 mr-2"></i>Urgent Complaints
        </h2>
        <?php if (empty($urgentComplaints)): ?>
        <div class="text-center py-8 text-gray-500">
            <i class="fas fa-check-circle text-4xl mb-3 text-green-500"></i>
            <p>No urgent complaints!</p>
        </div>
        <?php else: ?>
        <div class="space-y-3">
            <?php foreach ($urgentComplaints as $complaint): ?>
            <a href="<?= BASE_URL ?>/admin/complaint-detail.php?id=<?= $complaint['id'] ?>" 
               class="block p-3 bg-red-50 border border-red-200 rounded-lg hover:bg-red-100 transition">
                <p class="font-medium text-gray-800 line-clamp-1"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></p>
                <p class="text-sm text-gray-500 mt-1">
                    <?= sanitize($complaint['branch_name']) ?> • <?= formatDate($complaint['created_at']) ?>
                </p>
            </a>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Recent Complaints -->
<div class="bg-white rounded-xl shadow-lg overflow-hidden">
    <div class="p-6 border-b flex items-center justify-between">
        <h2 class="text-lg font-semibold text-gray-800">Recent Complaints</h2>
        <a href="<?= BASE_URL ?>/admin/complaints.php" class="text-primary hover:underline text-sm">View All</a>
    </div>
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ticket</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Issue</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden md:table-cell">Customer</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase hidden lg:table-cell">Branch</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Priority</th>
                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Action</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($recentComplaints as $complaint): ?>
                <tr class="hover:bg-gray-50 transition">
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="font-mono text-sm text-primary"><?= sanitize($complaint['ticket_number']) ?></span>
                    </td>
                    <td class="px-6 py-4">
                        <p class="text-gray-800 font-medium line-clamp-1 max-w-xs"><?= sanitize($complaint['issue_title'] ?? 'Complaint') ?></p>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden md:table-cell">
                        <span class="text-gray-600"><?= sanitize($complaint['customer_whatsapp']) ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap hidden lg:table-cell">
                        <span class="text-gray-600"><?= sanitize($complaint['branch_name']) ?></span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getStatusBadgeClass($complaint['status']) ?>">
                            <?= ucfirst(str_replace('_', ' ', $complaint['status'])) ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= getPriorityBadgeClass($complaint['priority']) ?>">
                            <?= ucfirst($complaint['priority']) ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-right">
                        <a href="<?= BASE_URL ?>/admin/complaint-detail.php?id=<?= $complaint['id'] ?>" 
                           class="text-primary hover:text-secondary font-medium text-sm">
                            Manage <i class="fas fa-arrow-right ml-1"></i>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
